//------------------------------------------------------------------------------
// File: svr_cmdupgrades.cs
// This file contains the server methods and data for the command upgrades. Note
// that current upgrade system is assuming that there is only one client.
// Author: Matthew Rudge
//------------------------------------------------------------------------------

// Starting values for upgrades
$CSUP_VALUES[$CSUP_GUN]             = false;
$CSUP_VALUES[$CSUP_AXE]             = false;
$CSUP_VALUES[$CSUP_PICK]            = false;
$CSUP_VALUES[$CSUP_BLDG]            = false; 
$CSUP_VALUES[$CSUP_CROPCORN]        = false;
$CSUP_VALUES[$CSUP_CROPPUMP]        = false;
$CSUP_VALUES[$CSUP_CROPFLWR]        = false; // OPTIONAL
$CSUP_VALUES[$CSUP_ANIMPIG]         = false;
$CSUP_VALUES[$CSUP_ANIMCOW]         = false;
$CSUP_VALUES[$CSUP_ANIMSHEEP]       = false; // OPTIONAL
$CSUP_VALUES[$CSUP_TRACK]           = false;
$CSUP_VALUES[$CSUP_FARMPRODUCTION]  = false; // HERO-BASED
$CSUP_VALUES[$CSUP_TAXPRODUCTION]   = false; // HERO-BASED
$CSUP_VALUES[$CSUP_MINEPRODUCTION]  = false; // HERO-BASED

////////////////////////////////////////////////////////////////////////////////
//! Returns the number of command upgrades, active or not
//! \retval int Number of command upgrades
////////////////////////////////////////////////////////////////////////////////
function csGetCmdUpgradeCount()
{
   return $CSUP_COUNT;
}

////////////////////////////////////////////////////////////////////////////////
//! Used to set the command upgrade
//! \param %client Client that is receiving upgrade (does not affect server)
//! \param %upgrade Upgrade
//! \param %val Value for upgrade
////////////////////////////////////////////////////////////////////////////////
function csSetCmdUpgrade(%client, %upgrade, %val)
{
   $CSUP_VALUES[%upgrade] = true;
   $CSUP_VALUES_DATA[%upgrade] = %val;
   commandToClient(%client, 'SetCmdUpgrade', %upgrade, true);
   
   // Switch on upgrade
   switch(%upgrade) {
      case $CSUP_GUN:
         // Break value into words
         %strength = getWord(%val, 0);
         %accuracy = getWord(%val, 1);
         osSetModifierValue($OSMOD_WEAPONSTR, %strength);
         osSetModifierValue($OSMOD_WEAPONACC, %accuracy);
      case $CSUP_AXE:
         osSetModifierValue($OSMOD_AXEUPGRADE, %val);
      case $CSUP_PICK:
         osSetModifierValue($OSMOD_PICKUPGRADE, %val);
      case $CSUP_BLDG:
         osSetModifierValue($OSMOD_HITPOINTS, %val);
      case $CSUP_TRACK:
         osSetModifierValue($OSMOD_TRAINPROD, %val);
         csLayTracks();
      case $CSUP_FARMPRODUCTION:
         setSpecialProduction("FarmData", $Resource::Food, %val);
      case $CSUP_TAXPRODUCTION:
         setSpecialProduction();
      case $CSUP_MINEPRODUCTION:
         setSpecialProduction("GoldMineData", $Resource::Gold, %val);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Resets all command upgrades
//! \param %client Client to reset for (this does not affect server)
////////////////////////////////////////////////////////////////////////////////
function csResetCmdUpgrades(%client)
{
   for(%i = 0; %i < $CSUP_COUNT; %i++) {
      $CSUP_VALUES[%i]        = false;
      $CSUP_VALUES_DATA[%i]   = "";
   }
   
   clearSpecialProduction();
   
   commandToClient(%client, 'ResetUpgrades');
}

////////////////////////////////////////////////////////////////////////////////
//! Tests whether an upgrade has been purchased or not
//! \param %upgrade Upgrade to check ($CSUP_GUN, $CSUP_AXE, etc.)
//! \retval bool True if the upgrade was purchased
////////////////////////////////////////////////////////////////////////////////
function csIsCmdUpgradePurchased(%upgrade)
{
   return $CSUP_VALUES[%upgrade];
}

////////////////////////////////////////////////////////////////////////////////
//! Purchases an upgrade
//! \param %client Client that purchased upgrade
//! \param %cmdData CommandData datablock that issued purchase
//! \param %upgrade Upgrade to purchase ($CSUP_GUN, $CSUP_AXE, etc.)
//! \param %value Value for upgrade
//! \param %index Button index for command response
////////////////////////////////////////////////////////////////////////////////
function csPurchaseCmdUpgrade(%client, %cmdData, %upgrade, %value, %index)
{
   // Purchase upgrade
   if(!%cmdData.canPurchaseCommand()) {
      notifyClientCmdBtnStatus(%client, %index, false);
      return;
   }
   %cmdData.purchaseCommand();
   
   // Update purchase
   csSetCmdUpgrade(%client, %upgrade, %value);
   notifyClientCmdBtnStatus(%client, %index, true);
}

////////////////////////////////////////////////////////////////////////////////
//! Command from a client to purchase an upgrade
//! \param %client Client to purchase upgrade from
//! \param %param Upgrade information (cmdData, upgrade type, and value in order)
////////////////////////////////////////////////////////////////////////////////
function serverCmdPurchaseUpgrade(%client, %param)
{
   %cmdData = getWord(%param, 0);
   %upgrade = getWord(%param, 1);
   %value   = getWord(%param, 2);
   %index   = getWord(%param, 3);
   if(getWordCount(%param) > 4) {
      %value = %value SPC getWord(%param, 3);
      %index = getWord(%param, 4);
   }
   if(!csIsCmdUpgradePurchased(%upgrade)) {
      csPurchaseCmdUpgrade(%client, %cmdData, %upgrade, %value, %index);
   }
   else {
      notifyClientCmdBtnStatus(%client, %index, false);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Executed when roads are upgraded to tracks
////////////////////////////////////////////////////////////////////////////////
function csLayTracks()
{   
   // Go through track objects and swap textures
   %i = 0;
   %trackName = "Track" @ %i;
   while(isObject(%trackName)) {
      %trackName.setSkinName("train");
      %i++;
      %trackName = "Track" @ %i;
   }
   
   echo("Stopped loading track textures at object: " @ %trackName @ ".");
   
   // Spawn train
   slgTrain_PickTrain();
   //slgTrain_ActivateTrain($TrainType::Default);
   
   // after the train tracks are placed, award the railroad badge
   tsAwardBadge("badge_railroad");
}

// End svr_cmdupgrades.cs
